from perfact.daemondatasources.datasourceProvider import createDS
from flask import Flask, request, Response
from threading import Thread
from waitress import serve
from time import sleep
import unittest
import json


class TestOData(unittest.TestCase):
    def setUp(self):
        super(unittest.TestCase, self).__init__()
        self.test_ip = '127.0.0.1'
        self.test_port = 1337
        self.csrf_token = "I'm a dummy token"
        self.csrf_cookie_name = "csrfcookie"
        self.csrf_cookie_value = "12345"
        self.csrf_cookie = "{name}={value}".format(
            name=self.csrf_cookie_name,
            value=self.csrf_cookie_value,
        )
        self.app = Flask(__name__)

        self.app.add_url_rule('/sub/domain', view_func=self.token_fetch_url)
        self.app.add_url_rule('/test', view_func=self.read_test)

        self.flask_thread = Thread(
            target=serve,
            kwargs={
                'app': self.app,
                'listen': '{ip}:{port}'.format(
                    ip=self.test_ip,
                    port=self.test_port,
                )
            },
        )
        self.flask_thread.daemon = True
        self.flask_thread.start()

        # Give Flask some time to start up
        sleep(3)

    def token_fetch_url(self):
        """ This function is setup at Flask for being called when fetching a
        token.
        We require the connection to ask for a token by setting the
        header 'x-csrf-token' to fetch
        """
        print('Received token request')
        print(request.headers)
        print("\n\n\n")

        if 'X-Csrf-Token' not in request.headers.keys():
            return Response(
                status=400,
            )

        if not 'fetch' == request.headers.get('x-csrf-token'):
            return Response(
                status=400,
            )

        return Response(
            status=200,
            headers={
                'x-csrf-token': self.csrf_token,
                'Set-Cookie': '{cookie}; Domain={ip}; Path=/'.format(
                    ip=self.test_ip,
                    cookie=self.csrf_cookie,
                ),
            },
        )

    def read_test(self):
        """ This function is setup at Flask for being called when reading data.
        We act like we need csrf-validation here and let the test run into an
        error when the connection failes to play the csrf-game with us
        """
        print('Received read request')
        print(request.headers)
        print("\n\n\n")

        if 'X-Csrf-Token' not in request.headers.keys():
            return {"error": True}

        if self.csrf_token != request.headers.get('x-csrf-token'):
            return {"error": True}

        if self.csrf_cookie_name not in request.cookies.keys():
            return {"error": True}

        cookie_value = request.cookies.get(self.csrf_cookie_name)
        if cookie_value != self.csrf_cookie_value:
            return {"error": True}

        return json.dumps({"foo": "bar"})

    def test_uncomplete_csrf_config(self):
        """ One of the keys in csrf is None here. This has to cause
        odata.is_csrf_configured to be False
        """
        config = {
            'base_url': 'http://{ip}:{port}'.format(
                        ip=self.test_ip,
                        port=self.test_port,
                    ),
            'csrf': {
                'token_fetch_url': None,
                'token_header': 'x-csrf-token',
                'token_cmd': 'fetch',
            }
        }

        odata_connection = createDS(
            classname='odata.OData',
            config=config,
        )

        self.assertFalse(odata_connection._is_csrf_configured())

    def test_odata_read(self):
        config = {
            'base_url': 'http://{ip}:{port}'.format(
                        ip=self.test_ip,
                        port=self.test_port,
                    ),
            'csrf': {
                'token_fetch_url': 'http://{ip}:{port}/sub/domain'.format(
                        ip=self.test_ip,
                        port=self.test_port,
                    ),
                'token_header': 'x-csrf-token',
                'token_cmd': 'fetch',
            }
        }

        odata_connection = createDS(
            classname='odata.OData',
            config=config,
        )
        odata_sim_response = odata_connection.read('/test')

        self.assertFalse("error" in odata_sim_response)

        self.assertTrue(odata_connection._is_csrf_configured())
        self.assertDictEqual(
            odata_sim_response,
            {'foo': 'bar'},
        )
